/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.io.input;

import com.google.common.base.Stopwatch;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.QueueInputStream;
import org.apache.commons.io.output.QueueOutputStream;
import org.apache.commons.lang3.StringUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

public class QueueInputStreamTest {
    public static Stream<Arguments> inputData() {
        return Stream.of(Arguments.of((Object[])new Object[]{""}), Arguments.of((Object[])new Object[]{"1"}), Arguments.of((Object[])new Object[]{"12"}), Arguments.of((Object[])new Object[]{"1234"}), Arguments.of((Object[])new Object[]{"12345678"}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)4095)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)4096)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)4097)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)8191)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)8192)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)8193)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)32768)}));
    }

    private int defaultBufferSize() {
        return 8192;
    }

    private String readUnbuffered(InputStream inputStream) throws IOException {
        return this.readUnbuffered(inputStream, Integer.MAX_VALUE);
    }

    private String readUnbuffered(InputStream inputStream, int maxBytes) throws IOException {
        if (maxBytes == 0) {
            return "";
        }
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        int n = -1;
        while ((n = inputStream.read()) != -1) {
            byteArrayOutputStream.write(n);
            if (byteArrayOutputStream.size() < maxBytes) continue;
            break;
        }
        return byteArrayOutputStream.toString(StandardCharsets.UTF_8.name());
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    public void testAvailableAfterClose(String inputData) throws IOException {
        QueueInputStream shadow;
        LinkedBlockingQueue<Integer> queue = new LinkedBlockingQueue<Integer>();
        try (QueueInputStream inputStream = new QueueInputStream(queue);){
            shadow = inputStream;
        }
        Assertions.assertEquals((int)0, (int)shadow.available());
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    public void testAvailableAfterOpen(String inputData) throws IOException {
        LinkedBlockingQueue<Integer> queue = new LinkedBlockingQueue<Integer>();
        try (QueueInputStream inputStream = new QueueInputStream(queue);){
            Assertions.assertEquals((int)0, (int)inputStream.available());
            IOUtils.toString((InputStream)inputStream, StandardCharsets.UTF_8);
            Assertions.assertEquals((int)0, (int)inputStream.available());
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    public void testBufferedReads(String inputData) throws IOException {
        LinkedBlockingQueue<Integer> queue = new LinkedBlockingQueue<Integer>();
        try (BufferedInputStream inputStream = new BufferedInputStream(new QueueInputStream(queue));
             QueueOutputStream outputStream = new QueueOutputStream(queue);){
            outputStream.write(inputData.getBytes(StandardCharsets.UTF_8));
            String actualData = IOUtils.toString((InputStream)inputStream, StandardCharsets.UTF_8);
            Assertions.assertEquals((Object)inputData, (Object)actualData);
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    public void testBufferedReadWrite(String inputData) throws IOException {
        LinkedBlockingQueue<Integer> queue = new LinkedBlockingQueue<Integer>();
        try (BufferedInputStream inputStream = new BufferedInputStream(new QueueInputStream(queue));
             BufferedOutputStream outputStream = new BufferedOutputStream(new QueueOutputStream(queue), this.defaultBufferSize());){
            outputStream.write(inputData.getBytes(StandardCharsets.UTF_8));
            outputStream.flush();
            String dataCopy = IOUtils.toString((InputStream)inputStream, StandardCharsets.UTF_8);
            Assertions.assertEquals((Object)inputData, (Object)dataCopy);
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    public void testBufferedWrites(String inputData) throws IOException {
        LinkedBlockingQueue<Integer> queue = new LinkedBlockingQueue<Integer>();
        try (QueueInputStream inputStream = new QueueInputStream(queue);
             BufferedOutputStream outputStream = new BufferedOutputStream(new QueueOutputStream(queue), this.defaultBufferSize());){
            outputStream.write(inputData.getBytes(StandardCharsets.UTF_8));
            outputStream.flush();
            String actualData = this.readUnbuffered(inputStream);
            Assertions.assertEquals((Object)inputData, (Object)actualData);
        }
    }

    @Test
    public void testInvalidArguments() {
        Assertions.assertThrows(NullPointerException.class, () -> new QueueInputStream(null), (String)"queue is required");
        Assertions.assertThrows(IllegalArgumentException.class, () -> QueueInputStream.builder().setTimeout(Duration.ofMillis(-1L)).get(), (String)"waitTime must not be negative");
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    public void testReadAfterClose(String inputData) throws IOException {
        QueueInputStream shadow;
        LinkedBlockingQueue<Integer> queue = new LinkedBlockingQueue<Integer>();
        try (QueueInputStream inputStream = new QueueInputStream(queue);){
            shadow = inputStream;
        }
        Assertions.assertEquals((int)-1, (int)((InputStream)shadow).read());
    }

    @Test
    public void testResetArguments() throws IOException {
        try (QueueInputStream queueInputStream = QueueInputStream.builder().setTimeout(null).get();){
            Assertions.assertEquals((Object)Duration.ZERO, (Object)queueInputStream.getTimeout());
            Assertions.assertEquals((int)0, (int)queueInputStream.getBlockingQueue().size());
        }
        queueInputStream = QueueInputStream.builder().setBlockingQueue(null).get();
        try {
            Assertions.assertEquals((Object)Duration.ZERO, (Object)queueInputStream.getTimeout());
            Assertions.assertEquals((int)0, (int)queueInputStream.getBlockingQueue().size());
        }
        finally {
            if (queueInputStream != null) {
                queueInputStream.close();
            }
        }
    }

    @Test
    @DisplayName(value="If read is interrupted while waiting, then exception is thrown")
    public void testTimeoutInterrupted() throws Exception {
        try (QueueInputStream inputStream = QueueInputStream.builder().setTimeout(Duration.ofMinutes(2L)).get();
             QueueOutputStream outputStream = inputStream.newQueueOutputStream();){
            AtomicBoolean result = new AtomicBoolean();
            CountDownLatch latch = new CountDownLatch(1);
            Thread thread = new Thread(() -> {
                Assertions.assertThrows(IllegalStateException.class, () -> this.readUnbuffered(inputStream, 3));
                Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
                result.set(true);
                latch.countDown();
            });
            thread.setDaemon(true);
            thread.start();
            thread.interrupt();
            latch.await(500L, TimeUnit.MILLISECONDS);
            Assertions.assertTrue((boolean)result.get());
        }
    }

    @Test
    @DisplayName(value="If data is not available in queue, then read will wait until wait time elapses")
    public void testTimeoutUnavailableData() throws IOException {
        try (QueueInputStream inputStream = QueueInputStream.builder().setTimeout(Duration.ofMillis(500L)).get();
             QueueOutputStream outputStream = inputStream.newQueueOutputStream();){
            Stopwatch stopwatch = Stopwatch.createStarted();
            String actualData = (String)Assertions.assertTimeout((Duration)Duration.ofSeconds(1L), () -> this.readUnbuffered(inputStream, 3));
            stopwatch.stop();
            Assertions.assertEquals((Object)"", (Object)actualData);
            Assertions.assertTrue((stopwatch.elapsed(TimeUnit.MILLISECONDS) >= 500L ? 1 : 0) != 0, () -> stopwatch.toString());
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    public void testUnbufferedReadWrite(String inputData) throws IOException {
        try (QueueInputStream inputStream = new QueueInputStream();
             QueueOutputStream outputStream = inputStream.newQueueOutputStream();){
            this.writeUnbuffered(outputStream, inputData);
            String actualData = this.readUnbuffered(inputStream);
            Assertions.assertEquals((Object)inputData, (Object)actualData);
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    public void testUnbufferedReadWriteWithTimeout(String inputData) throws IOException {
        Duration timeout = Duration.ofMinutes(2L);
        try (QueueInputStream inputStream = QueueInputStream.builder().setTimeout(timeout).get();
             QueueOutputStream outputStream = inputStream.newQueueOutputStream();){
            Assertions.assertEquals((Object)timeout, (Object)inputStream.getTimeout());
            this.writeUnbuffered(outputStream, inputData);
            String actualData = (String)Assertions.assertTimeout((Duration)Duration.ofSeconds(1L), () -> this.readUnbuffered(inputStream, inputData.length()));
            Assertions.assertEquals((Object)inputData, (Object)actualData);
        }
    }

    private void writeUnbuffered(QueueOutputStream outputStream, String inputData) throws IOException {
        byte[] bytes = inputData.getBytes(StandardCharsets.UTF_8);
        outputStream.write(bytes, 0, bytes.length);
    }
}

